#!/usr/bin/env python3
"""
Demo: Enhanced Video Recording with Audio Integration

This script demonstrates the new enhanced video recording capabilities
that create videos with integrated audio from team communications.
"""

import argparse
import json
import time
from pathlib import Path
from eval import MultiProviderEvaluator

def demo_enhanced_video():
    """Demonstrate enhanced video recording with different settings."""
    
    print("🎥 Enhanced Video Recording Demo")
    print("=" * 50)
    
    # Demo 1: Standard video (for comparison)
    print("\n📹 Demo 1: Standard Video Recording")
    print("-" * 30)
    
    evaluator_standard = MultiProviderEvaluator(
        difficulty="normal",
        seed_index=0,
        max_rounds=8,
        api_provider="qwen",
        input_mode="video",
        save_media=True,
        enhanced_video=False,  # Standard video
        video_fps=1.0
    )
    
    print(f"Standard setup - FPS: {evaluator_standard.video_fps}")
    print(f"Enhanced video: {evaluator_standard.enhanced_video}")
    
    # Run a short evaluation
    try:
        print("Running standard video evaluation...")
        results_standard = evaluator_standard.run_evaluation()
        print(f"✅ Standard video completed: {results_standard['final_stats']['final_score_normalized']:.1f}/100")
        
        # Save results
        results_file = evaluator_standard.save_results()
        print(f"📄 Results saved: {results_file}")
        
    except Exception as e:
        print(f"❌ Standard video demo failed: {e}")
    finally:
        evaluator_standard.close()
    
    print("\n" + "="*50)
    
    # Demo 2: Enhanced video with audio integration
    print("\n🎵 Demo 2: Enhanced Video with Audio Integration")
    print("-" * 40)
    
    evaluator_enhanced = MultiProviderEvaluator(
        difficulty="normal",
        seed_index=1,  # Different seed for variety
        max_rounds=8,
        api_provider="qwen",
        input_mode="video",
        save_media=True,
        enhanced_video=True,  # Enhanced video with audio
        video_fps=0.5,  # Slower FPS for more audio time
        audio_duration_per_frame=4.0  # 4 seconds of audio per frame
    )
    
    print(f"Enhanced setup - FPS: {evaluator_enhanced.video_fps}")
    print(f"Enhanced video: {evaluator_enhanced.enhanced_video}")
    print(f"Audio per frame: {evaluator_enhanced.audio_duration_per_frame}s")
    
    # Check video status
    video_status = evaluator_enhanced.env.get_video_status()
    print(f"Video system status: {json.dumps(video_status, indent=2)}")
    
    # Run enhanced evaluation
    try:
        print("Running enhanced video evaluation...")
        results_enhanced = evaluator_enhanced.run_evaluation()
        print(f"✅ Enhanced video completed: {results_enhanced['final_stats']['final_score_normalized']:.1f}/100")
        
        # Save results
        results_file = evaluator_enhanced.save_results()
        print(f"📄 Results saved: {results_file}")
        
        # Display enhanced video info
        if evaluator_enhanced.save_media:
            video_files = list(evaluator_enhanced.videos_dir.glob("*.mp4"))
            json_files = list(evaluator_enhanced.videos_dir.glob("*.json"))
            
            print(f"\n🎥 Generated files in {evaluator_enhanced.videos_dir}:")
            for video_file in video_files:
                size_mb = video_file.stat().st_size / (1024 * 1024)
                print(f"  📹 {video_file.name} ({size_mb:.1f} MB)")
            
            for json_file in json_files:
                if json_file.stem.endswith("_enhanced"):
                    print(f"  📋 {json_file.name} (enhanced metadata)")
                    
                    # Show metadata sample
                    try:
                        with open(json_file, 'r') as f:
                            metadata = json.load(f)
                        
                        print(f"    - Frames: {metadata.get('frames_count', 0)}")
                        print(f"    - Audio events: {metadata.get('audio_events_count', 0)}")
                        print(f"    - Successful audio integrations: {metadata.get('successful_audio_integrations', 0)}")
                        print(f"    - Video duration: {metadata.get('video_duration', 0):.1f}s")
                        
                    except Exception as e:
                        print(f"    - Could not read metadata: {e}")
        
    except Exception as e:
        print(f"❌ Enhanced video demo failed: {e}")
    finally:
        evaluator_enhanced.close()
    
    print("\n" + "="*50)
    
    # Demo 3: Very slow video for maximum audio integration
    print("\n🐌 Demo 3: Ultra-Slow Video for Maximum Audio")
    print("-" * 35)
    
    evaluator_ultra = MultiProviderEvaluator(
        difficulty="normal",
        seed_index=2,
        max_rounds=6,  # Fewer rounds since it will be slow
        api_provider="qwen",
        input_mode="video",
        save_media=True,
        enhanced_video=True,
        video_fps=0.25,  # Very slow - 4 seconds per frame
        audio_duration_per_frame=6.0  # 6 seconds of audio per frame
    )
    
    print(f"Ultra-slow setup - FPS: {evaluator_ultra.video_fps}")
    print(f"Audio per frame: {evaluator_ultra.audio_duration_per_frame}s")
    print("This will create a very slow video with lots of time for audio!")
    
    try:
        print("Running ultra-slow enhanced video evaluation...")
        results_ultra = evaluator_ultra.run_evaluation()
        print(f"✅ Ultra-slow video completed: {results_ultra['final_stats']['final_score_normalized']:.1f}/100")
        
        # Save results
        results_file = evaluator_ultra.save_results()
        print(f"📄 Results saved: {results_file}")
        
    except Exception as e:
        print(f"❌ Ultra-slow video demo failed: {e}")
    finally:
        evaluator_ultra.close()
    
    print("\n🎉 Enhanced Video Demo Complete!")
    print("\nCompare the generated videos to see the difference:")
    print("- Standard video: No integrated audio")
    print("- Enhanced video: Integrated team communications")  
    print("- Ultra-slow video: Maximum audio integration time")


def demo_batch_processing():
    """Demonstrate batch processing of existing media files into enhanced videos."""
    
    print("\n🔄 Batch Processing Demo")
    print("-" * 25)
    
    # Look for existing evaluation results
    outputs_dir = Path("outputs")
    if not outputs_dir.exists():
        print("⚠️ No outputs directory found. Run a regular evaluation first.")
        return
    
    # Find recent evaluation directories
    eval_dirs = [d for d in outputs_dir.iterdir() if d.is_dir()]
    if not eval_dirs:
        print("⚠️ No evaluation directories found.")
        return
    
    # Use the most recent directory
    latest_dir = max(eval_dirs, key=lambda d: d.stat().st_mtime)
    print(f"📁 Using latest evaluation: {latest_dir.name}")
    
    # Check for media files
    images_dir = latest_dir / "images"
    videos_dir = latest_dir / "videos"
    
    if not images_dir.exists():
        print("⚠️ No images directory found in evaluation results.")
        return
    
    # Find image files and video metadata
    image_files = list(images_dir.glob("*.jpg"))
    metadata_files = list(videos_dir.glob("*.json")) if videos_dir.exists() else []
    
    if not image_files:
        print("⚠️ No image files found.")
        return
    
    print(f"📸 Found {len(image_files)} image files")
    print(f"📋 Found {len(metadata_files)} metadata files")
    
    if metadata_files:
        # Use batch processing to create enhanced video
        from enhanced_video_recorder import create_enhanced_video_from_files
        
        # Use the first metadata file that looks like it has audio events
        audio_metadata_file = None
        for metadata_file in metadata_files:
            try:
                with open(metadata_file, 'r') as f:
                    data = json.load(f)
                    if data.get('audio_events'):
                        audio_metadata_file = metadata_file
                        break
            except:
                continue
        
        if audio_metadata_file:
            output_path = videos_dir / "batch_enhanced_video.mp4"
            print(f"🎬 Creating enhanced video from batch processing...")
            print(f"   Input: {len(image_files)} frames from {images_dir}")
            print(f"   Audio: {audio_metadata_file.name}")
            print(f"   Output: {output_path}")
            
            success, message = create_enhanced_video_from_files(
                str(images_dir),
                str(audio_metadata_file),
                str(output_path),
                fps=0.5
            )
            
            if success:
                print(f"✅ {message}")
                size_mb = output_path.stat().st_size / (1024 * 1024)
                print(f"📹 Enhanced video size: {size_mb:.1f} MB")
            else:
                print(f"❌ Batch processing failed: {message}")
        else:
            print("⚠️ No metadata files with audio events found.")
    else:
        print("⚠️ No metadata files found for batch processing.")


def main():
    """Main demo function with command line options."""
    parser = argparse.ArgumentParser(description="Enhanced Video Recording Demo")
    parser.add_argument("--demo", type=str, choices=["enhanced", "batch", "both"], 
                        default="both", help="Which demo to run")
    parser.add_argument("--api_provider", type=str, default="qwen", 
                        choices=["qwen", "openai"], help="API provider to use")
    
    args = parser.parse_args()
    
    print(f"🚀 Enhanced Video Demo - Provider: {args.api_provider.upper()}")
    
    if args.demo in ["enhanced", "both"]:
        demo_enhanced_video()
    
    if args.demo in ["batch", "both"]:
        demo_batch_processing()
    
    print("\n✨ Demo complete! Check the output directories for your enhanced videos.")


if __name__ == "__main__":
    main() 